import numpy as np
import torch

class EarlyStopping:
    """Stops the training early if a monitored metric has not improved after a given patience."""
    def __init__(self, patience=7, verbose=False, delta=0, 
                 monitor_metric_name="val_loss", mode="min", trace_func=print):
        """
        Args:
            patience (int): How long to wait after last time the monitored metric improved.
            verbose (bool): If True, prints a message for each improvement.
            delta (float): Minimum change in the monitored quantity to qualify as an improvement.
            monitor_metric_name (str): Name of the metric to monitor.
            mode (str): One of 'min' or 'max'. In 'min' mode, training stops when the metric stops decreasing; 
                        in 'max' mode it stops when the metric stops increasing.
            trace_func (function): Function to use for printing messages.
        """
        self.patience = patience
        self.verbose = verbose
        self.counter = 0
        self.best_score = None
        self.early_stop = False
        self.delta = delta
        self.mode = mode.lower()
        self.trace_func = trace_func
        self.monitor_metric_name = monitor_metric_name

        if self.mode not in ['min', 'max']:
            raise ValueError("mode must be 'min' or 'max'.")

    def __call__(self, current_metric_value):
        """
        Call this at the end of each epoch to check if training should stop.
        """
        score = current_metric_value

        if self.best_score is None:
            self.best_score = score
            if self.verbose:
                self.trace_func(f"EarlyStopping initialized. Monitored metric '{self.monitor_metric_name}' set to {self.best_score:.6f}")
            return

        improvement_occurred = False
        if self.mode == "min":
            # An improvement is a score smaller than the best score by at least delta.
            if score < self.best_score - self.delta:
                improvement_occurred = True
                if self.verbose:
                    self.trace_func(f"{self.monitor_metric_name} decreased ({self.best_score:.6f} --> {score:.6f}). Resetting counter.")
        
        elif self.mode == "max":
            # An improvement is a score larger than the best score by at least delta.
            if score > self.best_score + self.delta:
                improvement_occurred = True
                if self.verbose:
                    self.trace_func(f"{self.monitor_metric_name} increased ({self.best_score:.6f} --> {score:.6f}). Resetting counter.")

        if improvement_occurred:
            self.best_score = score
            self.counter = 0
        else:
            self.counter += 1
            if self.verbose:
                self.trace_func(f"EarlyStopping counter: {self.counter} out of {self.patience} "
                                f"({self.monitor_metric_name}: {score:.6f}, Best: {self.best_score:.6f})")

        if self.counter >= self.patience:
            self.early_stop = True
            if self.verbose:
                self.trace_func(f"Early stopping triggered: {self.monitor_metric_name} did not improve for {self.patience} epochs.")